<?php
require_once '../includes/config.php';
require_once '../includes/auth.php';
require_once '../includes/db.php';
require_once '../includes/paystack.php';
require_once '../includes/email.php';

$auth = getAuth();

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ../public/create-invoice.php');
    exit();
}

// Require login
$auth->requireLogin();
$user = $auth->getCurrentUser();

// Verify CSRF token
if (!$auth->verifyCSRFToken($_POST['csrf_token'] ?? '')) {
    $_SESSION['error_message'] = 'Invalid request. Please try again.';
    header('Location: ../public/create-invoice.php');
    exit();
}

// Sanitize and validate input
$invoiceNumber = $auth->sanitizeInput($_POST['invoice_number'] ?? '');
$clientName = $auth->sanitizeInput($_POST['client_name'] ?? '');
$clientEmail = $auth->sanitizeInput($_POST['client_email'] ?? '');
$dueDate = $auth->sanitizeInput($_POST['due_date'] ?? '');
$items = $_POST['items'] ?? [];

// Validation
$errors = [];

if (empty($invoiceNumber)) {
    $errors[] = 'Invoice number is required';
}

if (empty($clientName)) {
    $errors[] = 'Client name is required';
}

if (empty($clientEmail) || !$auth->validateEmail($clientEmail)) {
    $errors[] = 'Valid client email is required';
}

if (empty($dueDate)) {
    $errors[] = 'Due date is required';
}

if (empty($items) || !is_array($items)) {
    $errors[] = 'At least one invoice item is required';
}

// Validate items
$subtotal = 0;
$validItems = [];

foreach ($items as $item) {
    $description = $auth->sanitizeInput($item['description'] ?? '');
    $quantity = intval($item['quantity'] ?? 0);
    $unitPrice = floatval($item['unit_price'] ?? 0);
    $total = floatval($item['total'] ?? 0);
    
    if (empty($description)) {
        $errors[] = 'Item description is required';
        continue;
    }
    
    if ($quantity <= 0) {
        $errors[] = 'Item quantity must be greater than 0';
        continue;
    }
    
    if ($unitPrice < 0) {
        $errors[] = 'Item unit price cannot be negative';
        continue;
    }
    
    // Recalculate total to prevent tampering
    $calculatedTotal = $quantity * $unitPrice;
    
    $validItems[] = [
        'description' => $description,
        'quantity' => $quantity,
        'unit_price' => $unitPrice,
        'total' => $calculatedTotal
    ];
    
    $subtotal += $calculatedTotal;
}

if ($subtotal <= 0) {
    $errors[] = 'Invoice total must be greater than 0';
}

// If validation fails, redirect back with errors
if (!empty($errors)) {
    $_SESSION['error_message'] = implode('<br>', $errors);
    header('Location: ../public/create-invoice.php');
    exit();
}

$db = getDB();

try {
    // Start transaction
    $db->beginTransaction();
    
    // Calculate totals
    $taxAmount = 0; // No tax for now
    $totalAmount = $subtotal + $taxAmount;
    
    // Insert invoice
    $stmt = $db->prepare("INSERT INTO invoices (user_id, invoice_number, client_name, client_email, subtotal, tax_amount, total_amount, due_date, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'unpaid')");
    $stmt->bind_param("isssddds", $user['id'], $invoiceNumber, $clientName, $clientEmail, $subtotal, $taxAmount, $totalAmount, $dueDate);
    
    if (!$stmt->execute()) {
        throw new Exception("Failed to create invoice");
    }
    
    $invoiceId = $db->getLastInsertId();
    
    // Insert invoice items
    $stmt = $db->prepare("INSERT INTO invoice_items (invoice_id, item_name, quantity, unit_price, total_price) VALUES (?, ?, ?, ?, ?)");
    
    foreach ($validItems as $item) {
        $stmt->bind_param("isidd", $invoiceId, $item['description'], $item['quantity'], $item['unit_price'], $item['total']);
        if (!$stmt->execute()) {
            throw new Exception("Failed to add invoice item");
        }
    }
    
    // Create Paystack payment link (only if keys are configured)
    if (PAYSTACK_SECRET_KEY !== 'sk_test_your_actual_secret_key_here' && !empty(PAYSTACK_SECRET_KEY)) {
        $paystack = new PaystackAPI();
        $paymentData = [
            'email' => $clientEmail,
            'amount' => $totalAmount * 100, // Convert to kobo
            'reference' => 'inv_' . $invoiceId . '_' . time(),
            'callback_url' => SITE_URL . '/payment-success.php',
            'metadata' => [
                'invoice_id' => $invoiceId,
                'invoice_number' => $invoiceNumber,
                'client_name' => $clientName
            ]
        ];
        
        $paymentResult = $paystack->createPaymentLink($paymentData);
        
        if ($paymentResult['success']) {
            // Update invoice with payment link
            $stmt = $db->prepare("UPDATE invoices SET paystack_reference = ?, paystack_payment_link = ? WHERE id = ?");
            $stmt->bind_param("ssi", $paymentResult['reference'], $paymentResult['authorization_url'], $invoiceId);
            $stmt->execute();
            
            $paymentLink = $paymentResult['authorization_url'];
        } else {
            // Log error but don't fail invoice creation
            error_log("Paystack payment link creation failed: " . $paymentResult['message']);
            $paymentLink = SITE_URL . '/pay-invoice.php?id=' . $invoiceId;
        }
    } else {
        // Use dummy payment link when Paystack is not configured
        $paymentReference = 'inv_' . $invoiceId . '_' . time();
        $paymentLink = SITE_URL . '/pay-invoice.php?id=' . $invoiceId;
        
        $stmt = $db->prepare("UPDATE invoices SET paystack_reference = ?, paystack_payment_link = ? WHERE id = ?");
        $stmt->bind_param("ssi", $paymentReference, $paymentLink, $invoiceId);
        $stmt->execute();
    }
    
    // Commit transaction
    $db->commit();
    
    // Send invoice email (optional - skip if not configured)
    try {
        if (SMTP_USERNAME !== 'your_email@gmail.com' && !empty(SMTP_USERNAME)) {
            $emailSender = new EmailSender();
            $invoiceData = [
                'invoice_number' => $invoiceNumber,
                'sender_name' => $user['name'],
                'total_amount' => $totalAmount,
                'due_date' => date('F j, Y', strtotime($dueDate))
            ];
            
            $emailResult = $emailSender->sendInvoiceEmail($clientEmail, $clientName, $invoiceData, $paymentLink);
            
            if ($emailResult['success']) {
                $_SESSION['success_message'] = 'Invoice created successfully and sent to client!';
            } else {
                $_SESSION['success_message'] = 'Invoice created successfully! Email sending failed - you can resend from the invoice page.';
            }
        } else {
            $_SESSION['success_message'] = 'Invoice created successfully! (Email not configured)';
        }
    } catch (Exception $emailError) {
        error_log("Email sending error: " . $emailError->getMessage());
        $_SESSION['success_message'] = 'Invoice created successfully! (Email sending failed)';
    }
    
    header('Location: ../public/view-invoice.php?id=' . $invoiceId);
    
} catch (Exception $e) {
    // Rollback transaction
    $db->rollback();
    
    error_log("Invoice creation error: " . $e->getMessage());
    $_SESSION['error_message'] = 'Failed to create invoice. Please try again.';
    header('Location: ../public/create-invoice.php');
}

exit();
?>
