<?php
session_start();
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/auth.php';
require_once '../includes/email.php';
require_once '../includes/paystack.php';

// Check if user is logged in
$auth = new Auth();
if (!$auth->isLoggedIn()) {
    header('Location: ../public/login.php');
    exit;
}

// Verify CSRF token
if (!$auth->verifyCsrfToken($_POST['csrf_token'] ?? '')) {
    $_SESSION['error_message'] = 'Invalid request. Please try again.';
    header('Location: ../public/invoices.php');
    exit;
}

// Get invoice ID
$invoiceId = filter_input(INPUT_POST, 'invoice_id', FILTER_VALIDATE_INT);
if (!$invoiceId) {
    $_SESSION['error_message'] = 'Invalid invoice ID.';
    header('Location: ../public/invoices.php');
    exit;
}

try {
    $db = Database::getInstance()->getConnection();
    $userId = $_SESSION['user_id'];
    
    // Get invoice details with items
    $stmt = $db->prepare("
        SELECT i.*, 
               GROUP_CONCAT(
                   CONCAT(ii.item_name, '|', ii.quantity, '|', ii.unit_price, '|', ii.total_price) 
                   SEPARATOR ';;'
               ) as items
        FROM invoices i 
        LEFT JOIN invoice_items ii ON i.id = ii.invoice_id 
        WHERE i.id = ? AND i.user_id = ?
        GROUP BY i.id
    ");
    $stmt->bind_param("ii", $invoiceId, $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $invoice = $result->fetch_assoc();
    
    if (!$invoice) {
        $_SESSION['error_message'] = 'Invoice not found.';
        header('Location: ../public/invoices.php');
        exit;
    }
    
    // Get user details
    $stmt = $db->prepare("SELECT name, email FROM users WHERE id = ?");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $user = $stmt->get_result()->fetch_assoc();
    
    // Check if invoice already has a payment link
    $paymentLink = $invoice['paystack_payment_link'];
    
    // If no payment link exists, create one
    if (empty($paymentLink) || $paymentLink === SITE_URL . '/pay-invoice.php?id=' . $invoiceId) {
        // Create Paystack payment link if configured
        if (PAYSTACK_SECRET_KEY !== 'sk_test_your_actual_secret_key_here' && !empty(PAYSTACK_SECRET_KEY)) {
            $paystack = new PaystackAPI();
            $paymentData = [
                'email' => $invoice['client_email'],
                'amount' => $invoice['total_amount'] * 100, // Convert to kobo
                'reference' => 'inv_' . $invoiceId . '_' . time(),
                'callback_url' => SITE_URL . '/payment-success.php',
                'metadata' => [
                    'invoice_id' => $invoiceId,
                    'invoice_number' => $invoice['invoice_number'],
                    'client_name' => $invoice['client_name']
                ]
            ];
            
            $paymentResult = $paystack->createPaymentLink($paymentData);
            
            if ($paymentResult['success']) {
                // Update invoice with payment link
                $stmt = $db->prepare("UPDATE invoices SET paystack_reference = ?, paystack_payment_link = ? WHERE id = ?");
                $stmt->bind_param("ssi", $paymentResult['reference'], $paymentResult['authorization_url'], $invoiceId);
                $stmt->execute();
                
                $paymentLink = $paymentResult['authorization_url'];
            } else {
                error_log("Paystack payment link creation failed: " . $paymentResult['message']);
                $paymentLink = SITE_URL . '/pay-invoice.php?id=' . $invoiceId;
            }
        } else {
            // Use dummy payment link when Paystack is not configured
            $paymentLink = SITE_URL . '/pay-invoice.php?id=' . $invoiceId;
        }
    }
    
    // Send invoice email
    $emailSender = new EmailSender();
    $invoiceData = [
        'invoice_number' => $invoice['invoice_number'],
        'sender_name' => $user['name'],
        'total_amount' => $invoice['total_amount'],
        'due_date' => date('F j, Y', strtotime($invoice['due_date']))
    ];
    
    $emailResult = $emailSender->sendInvoiceEmail(
        $invoice['client_email'], 
        $invoice['client_name'], 
        $invoiceData, 
        $paymentLink
    );
    
    if ($emailResult['success']) {
        // Update invoice status to sent if it was draft
        if ($invoice['status'] === 'draft') {
            $stmt = $db->prepare("UPDATE invoices SET status = 'sent', sent_at = NOW() WHERE id = ?");
            $stmt->bind_param("i", $invoiceId);
            $stmt->execute();
        } else {
            // Update sent_at timestamp for resends
            $stmt = $db->prepare("UPDATE invoices SET sent_at = NOW() WHERE id = ?");
            $stmt->bind_param("i", $invoiceId);
            $stmt->execute();
        }
        
        $_SESSION['success_message'] = 'Invoice sent successfully to ' . $invoice['client_email'] . '!';
    } else {
        $_SESSION['error_message'] = 'Failed to send invoice email. Please try again.';
    }
    
} catch (Exception $e) {
    error_log("Send invoice error: " . $e->getMessage());
    $_SESSION['error_message'] = 'An error occurred while sending the invoice. Please try again.';
}

// Redirect back to invoice view or invoices list
$redirectUrl = isset($_POST['redirect_to']) ? $_POST['redirect_to'] : '../public/invoices.php';
header('Location: ' . $redirectUrl);
exit;
?>
