// SwiftBill NG JavaScript Application

// Global utilities
window.SwiftBill = {
    // Format currency
    formatCurrency: function(amount) {
        return '₦' + parseFloat(amount || 0).toLocaleString('en-NG', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
    },
    
    // Show toast notification
    showToast: function(message, type = 'info') {
        const toastContainer = document.getElementById('toast-container') || this.createToastContainer();
        const toast = this.createToast(message, type);
        toastContainer.appendChild(toast);
        
        // Show toast
        const bsToast = new bootstrap.Toast(toast);
        bsToast.show();
        
        // Remove from DOM after hidden
        toast.addEventListener('hidden.bs.toast', () => {
            toast.remove();
        });
    },
    
    // Create toast container if it doesn't exist
    createToastContainer: function() {
        const container = document.createElement('div');
        container.id = 'toast-container';
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '1055';
        document.body.appendChild(container);
        return container;
    },
    
    // Create toast element
    createToast: function(message, type) {
        const toast = document.createElement('div');
        toast.className = 'toast';
        toast.setAttribute('role', 'alert');
        
        const iconMap = {
            success: 'bi-check-circle-fill text-success',
            error: 'bi-exclamation-triangle-fill text-danger',
            warning: 'bi-exclamation-triangle-fill text-warning',
            info: 'bi-info-circle-fill text-info'
        };
        
        const icon = iconMap[type] || iconMap.info;
        
        toast.innerHTML = `
            <div class="toast-header">
                <i class="bi ${icon} me-2"></i>
                <strong class="me-auto">SwiftBill NG</strong>
                <button type="button" class="btn-close" data-bs-dismiss="toast"></button>
            </div>
            <div class="toast-body">
                ${message}
            </div>
        `;
        
        return toast;
    },
    
    // Confirm dialog
    confirm: function(message, callback) {
        if (confirm(message)) {
            callback();
        }
    },
    
    // Copy to clipboard
    copyToClipboard: function(text) {
        navigator.clipboard.writeText(text).then(() => {
            this.showToast('Copied to clipboard!', 'success');
        }).catch(() => {
            this.showToast('Failed to copy to clipboard', 'error');
        });
    },
    
    // Validate email
    validateEmail: function(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    },
    
    // Debounce function
    debounce: function(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
};

// Initialize application when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize tooltips
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function(tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
    
    // Initialize popovers
    const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
    popoverTriggerList.map(function(popoverTriggerEl) {
        return new bootstrap.Popover(popoverTriggerEl);
    });
    
    // Auto-hide alerts after 5 seconds
    const alerts = document.querySelectorAll('.alert:not(.alert-permanent)');
    alerts.forEach(alert => {
        setTimeout(() => {
            const bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        }, 5000);
    });
    
    // Add loading states to forms
    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const submitBtn = form.querySelector('button[type="submit"]');
            if (submitBtn && !submitBtn.disabled) {
                submitBtn.disabled = true;
                const originalText = submitBtn.textContent;
                submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Loading...';
                
                // Re-enable after 10 seconds as fallback
                setTimeout(() => {
                    submitBtn.disabled = false;
                    submitBtn.textContent = originalText;
                }, 10000);
            }
        });
    });
    
    // Add copy functionality to elements with data-copy attribute
    const copyElements = document.querySelectorAll('[data-copy]');
    copyElements.forEach(element => {
        element.style.cursor = 'pointer';
        element.addEventListener('click', function() {
            const textToCopy = this.getAttribute('data-copy') || this.textContent;
            SwiftBill.copyToClipboard(textToCopy);
        });
    });
    
    // Add confirmation to delete buttons
    const deleteButtons = document.querySelectorAll('[data-confirm]');
    deleteButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            const message = this.getAttribute('data-confirm') || 'Are you sure?';
            SwiftBill.confirm(message, () => {
                if (this.href) {
                    window.location.href = this.href;
                } else if (this.form) {
                    this.form.submit();
                }
            });
        });
    });
    
    // Auto-format currency inputs
    const currencyInputs = document.querySelectorAll('input[data-currency]');
    currencyInputs.forEach(input => {
        input.addEventListener('blur', function() {
            const value = parseFloat(this.value);
            if (!isNaN(value)) {
                this.value = value.toFixed(2);
            }
        });
    });
    
    // Add search functionality
    const searchInputs = document.querySelectorAll('input[data-search]');
    searchInputs.forEach(input => {
        const targetSelector = input.getAttribute('data-search');
        const searchFunction = SwiftBill.debounce(function() {
            const query = input.value.toLowerCase();
            const targets = document.querySelectorAll(targetSelector);
            
            targets.forEach(target => {
                const text = target.textContent.toLowerCase();
                const shouldShow = text.includes(query);
                target.style.display = shouldShow ? '' : 'none';
            });
        }, 300);
        
        input.addEventListener('input', searchFunction);
    });
    
    // Add smooth scrolling to anchor links
    const anchorLinks = document.querySelectorAll('a[href^="#"]');
    anchorLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            const targetId = this.getAttribute('href');
            const targetElement = document.querySelector(targetId);
            
            if (targetElement) {
                e.preventDefault();
                targetElement.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
    
    // Add fade-in animation to cards
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('fade-in-up');
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);
    
    const cards = document.querySelectorAll('.card');
    cards.forEach(card => {
        observer.observe(card);
    });
});

// Invoice-specific functionality
window.InvoiceManager = {
    // Calculate invoice totals
    calculateTotals: function(items) {
        const subtotal = items.reduce((sum, item) => {
            return sum + (parseFloat(item.quantity || 0) * parseFloat(item.unit_price || 0));
        }, 0);
        
        const tax = 0; // No tax for now
        const total = subtotal + tax;
        
        return { subtotal, tax, total };
    },
    
    // Validate invoice form
    validateInvoiceForm: function(formData) {
        const errors = [];
        
        if (!formData.client_name?.trim()) {
            errors.push('Client name is required');
        }
        
        if (!formData.client_email?.trim() || !SwiftBill.validateEmail(formData.client_email)) {
            errors.push('Valid client email is required');
        }
        
        if (!formData.due_date) {
            errors.push('Due date is required');
        }
        
        if (!formData.items || formData.items.length === 0) {
            errors.push('At least one invoice item is required');
        }
        
        formData.items?.forEach((item, index) => {
            if (!item.description?.trim()) {
                errors.push(`Item ${index + 1}: Description is required`);
            }
            
            if (!item.quantity || item.quantity <= 0) {
                errors.push(`Item ${index + 1}: Quantity must be greater than 0`);
            }
            
            if (item.unit_price === undefined || item.unit_price < 0) {
                errors.push(`Item ${index + 1}: Unit price cannot be negative`);
            }
        });
        
        return errors;
    },
    
    // Generate invoice preview
    generatePreview: function(formData) {
        const totals = this.calculateTotals(formData.items);
        
        return `
            <div class="invoice-preview">
                <div class="d-flex justify-content-between mb-4">
                    <div>
                        <h4>Invoice Preview</h4>
                        <p class="text-muted">Invoice #${formData.invoice_number}</p>
                    </div>
                    <div class="text-end">
                        <p><strong>Due Date:</strong> ${new Date(formData.due_date).toLocaleDateString()}</p>
                    </div>
                </div>
                
                <div class="row mb-4">
                    <div class="col-md-6">
                        <h6>Bill To:</h6>
                        <p class="mb-1"><strong>${formData.client_name}</strong></p>
                        <p class="text-muted">${formData.client_email}</p>
                    </div>
                </div>
                
                <div class="table-responsive mb-4">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Description</th>
                                <th>Qty</th>
                                <th>Price</th>
                                <th>Total</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${formData.items.map(item => `
                                <tr>
                                    <td>${item.description}</td>
                                    <td>${item.quantity}</td>
                                    <td>${SwiftBill.formatCurrency(item.unit_price)}</td>
                                    <td>${SwiftBill.formatCurrency(item.quantity * item.unit_price)}</td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                </div>
                
                <div class="row">
                    <div class="col-md-6 offset-md-6">
                        <div class="table-responsive">
                            <table class="table">
                                <tr>
                                    <td>Subtotal:</td>
                                    <td class="text-end">${SwiftBill.formatCurrency(totals.subtotal)}</td>
                                </tr>
                                <tr>
                                    <td>Tax:</td>
                                    <td class="text-end">${SwiftBill.formatCurrency(totals.tax)}</td>
                                </tr>
                                <tr class="fw-bold">
                                    <td>Total:</td>
                                    <td class="text-end">${SwiftBill.formatCurrency(totals.total)}</td>
                                </tr>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        `;
    }
};

// Payment functionality
window.PaymentManager = {
    // Initialize Paystack payment
    initializePayment: function(config) {
        if (typeof PaystackPop === 'undefined') {
            SwiftBill.showToast('Payment system not loaded. Please refresh the page.', 'error');
            return;
        }
        
        const handler = PaystackPop.setup({
            key: config.public_key,
            email: config.email,
            amount: config.amount * 100, // Convert to kobo
            currency: 'NGN',
            ref: config.reference,
            metadata: config.metadata || {},
            callback: function(response) {
                SwiftBill.showToast('Payment successful! Redirecting...', 'success');
                if (config.callback_url) {
                    window.location.href = config.callback_url + '?reference=' + response.reference;
                }
            },
            onClose: function() {
                SwiftBill.showToast('Payment cancelled', 'warning');
            }
        });
        
        handler.openIframe();
    }
};

// Export for use in other scripts
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { SwiftBill, InvoiceManager, PaymentManager };
}
