<?php
// Authentication and Security Functions
require_once 'config.php';
require_once 'db.php';

class Auth {
    private $db;
    
    public function __construct() {
        $this->db = getDB();
    }
    
    // Generate CSRF token
    public function generateCSRFToken() {
        $token = bin2hex(random_bytes(32));
        $_SESSION['csrf_token'] = $token;
        $_SESSION['csrf_token_time'] = time();
        return $token;
    }
    
    // Verify CSRF token
    public function verifyCSRFToken($token) {
        if (!isset($_SESSION['csrf_token']) || !isset($_SESSION['csrf_token_time'])) {
            return false;
        }
        
        // Check if token has expired
        if (time() - $_SESSION['csrf_token_time'] > CSRF_TOKEN_EXPIRE) {
            unset($_SESSION['csrf_token']);
            unset($_SESSION['csrf_token_time']);
            return false;
        }
        
        return hash_equals($_SESSION['csrf_token'], $token);
    }
    
    // Sanitize input to prevent XSS
    public function sanitizeInput($input) {
        return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
    }
    
    // Validate email format
    public function validateEmail($email) {
        return filter_var($email, FILTER_VALIDATE_EMAIL);
    }
    
    // Validate password strength
    public function validatePassword($password) {
        if (strlen($password) < PASSWORD_MIN_LENGTH) {
            return false;
        }
        
        // Check for at least one uppercase, lowercase, number, and special character
        if (!preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]/', $password)) {
            return false;
        }
        
        return true;
    }
    
    // Hash password
    public function hashPassword($password) {
        return password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
    }
    
    // Verify password
    public function verifyPassword($password, $hash) {
        return password_verify($password, $hash);
    }
    
    // Generate verification token
    public function generateVerificationToken() {
        return bin2hex(random_bytes(32));
    }
    
    // Register new user
    public function register($fullName, $email, $password) {
        try {
            // Check if email already exists
            $stmt = $this->db->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->bind_param("s", $email);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                return ['success' => false, 'message' => 'Email already exists'];
            }
            
            // Hash password and generate verification token
            $passwordHash = $this->hashPassword($password);
            $verificationToken = $this->generateVerificationToken();
            
            // Insert new user
            $stmt = $this->db->prepare("INSERT INTO users (full_name, email, password_hash, verification_token) VALUES (?, ?, ?, ?)");
            $stmt->bind_param("ssss", $fullName, $email, $passwordHash, $verificationToken);
            
            if ($stmt->execute()) {
                return [
                    'success' => true, 
                    'message' => 'Registration successful. Please check your email for verification.',
                    'user_id' => $this->db->getLastInsertId(),
                    'verification_token' => $verificationToken
                ];
            } else {
                return ['success' => false, 'message' => 'Registration failed. Please try again.'];
            }
            
        } catch (Exception $e) {
            error_log("Registration error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Registration failed. Please try again.'];
        }
    }
    
    // Login user
    public function login($email, $password) {
        try {
            $stmt = $this->db->prepare("SELECT id, full_name, password_hash, email_verified FROM users WHERE email = ?");
            $stmt->bind_param("s", $email);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows === 0) {
                return ['success' => false, 'message' => 'Invalid email or password'];
            }
            
            $user = $result->fetch_assoc();
            
            if (!$this->verifyPassword($password, $user['password_hash'])) {
                return ['success' => false, 'message' => 'Invalid email or password'];
            }
            
            if (!$user['email_verified']) {
                return ['success' => false, 'message' => 'Please verify your email before logging in'];
            }
            
            // Set session variables
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user_name'] = $user['full_name'];
            $_SESSION['user_email'] = $email;
            $_SESSION['login_time'] = time();
            
            return ['success' => true, 'message' => 'Login successful'];
            
        } catch (Exception $e) {
            error_log("Login error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Login failed. Please try again.'];
        }
    }
    
    // Check if user is logged in
    public function isLoggedIn() {
        if (!isset($_SESSION['user_id']) || !isset($_SESSION['login_time'])) {
            return false;
        }
        
        // Check if session has expired
        if (time() - $_SESSION['login_time'] > SESSION_EXPIRE) {
            $this->logout();
            return false;
        }
        
        return true;
    }
    
    // Get current user info
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        return [
            'id' => $_SESSION['user_id'],
            'name' => $_SESSION['user_name'],
            'email' => $_SESSION['user_email']
        ];
    }
    
    // Logout user
    public function logout() {
        session_unset();
        session_destroy();
    }
    
    // Verify email with token
    public function verifyEmail($token) {
        try {
            $stmt = $this->db->prepare("UPDATE users SET email_verified = 1, verification_token = NULL WHERE verification_token = ?");
            $stmt->bind_param("s", $token);
            
            if ($stmt->execute() && $stmt->affected_rows > 0) {
                return ['success' => true, 'message' => 'Email verified successfully'];
            } else {
                return ['success' => false, 'message' => 'Invalid verification token'];
            }
            
        } catch (Exception $e) {
            error_log("Email verification error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Verification failed. Please try again.'];
        }
    }
    
    // Generate password reset token
    public function generatePasswordResetToken($email) {
        try {
            $token = $this->generateVerificationToken();
            $expires = date('Y-m-d H:i:s', time() + 3600); // 1 hour
            
            $stmt = $this->db->prepare("UPDATE users SET reset_token = ?, reset_token_expires = ? WHERE email = ?");
            $stmt->bind_param("sss", $token, $expires, $email);
            
            if ($stmt->execute() && $stmt->affected_rows > 0) {
                return ['success' => true, 'token' => $token];
            } else {
                return ['success' => false, 'message' => 'Email not found'];
            }
            
        } catch (Exception $e) {
            error_log("Password reset token error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to generate reset token'];
        }
    }
    
    // Reset password with token
    public function resetPassword($token, $newPassword) {
        try {
            $stmt = $this->db->prepare("SELECT id FROM users WHERE reset_token = ? AND reset_token_expires > NOW()");
            $stmt->bind_param("s", $token);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows === 0) {
                return ['success' => false, 'message' => 'Invalid or expired reset token'];
            }
            
            $user = $result->fetch_assoc();
            $passwordHash = $this->hashPassword($newPassword);
            
            $stmt = $this->db->prepare("UPDATE users SET password_hash = ?, reset_token = NULL, reset_token_expires = NULL WHERE id = ?");
            $stmt->bind_param("si", $passwordHash, $user['id']);
            
            if ($stmt->execute()) {
                return ['success' => true, 'message' => 'Password reset successful'];
            } else {
                return ['success' => false, 'message' => 'Password reset failed'];
            }
            
        } catch (Exception $e) {
            error_log("Password reset error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Password reset failed'];
        }
    }
    
    // Require login (redirect if not logged in)
    public function requireLogin() {
        if (!$this->isLoggedIn()) {
            header('Location: login.php');
            exit();
        }
    }
}

// Helper function to get auth instance
function getAuth() {
    return new Auth();
}
?>
