<?php
require_once '../includes/config.php';
require_once '../includes/auth.php';

$auth = getAuth();
$page_title = 'Create Invoice';

// Require login
$auth->requireLogin();

$user = $auth->getCurrentUser();

// Generate CSRF token
$csrf_token = $auth->generateCSRFToken();

// Generate next invoice number
$db = getDB();
try {
    $stmt = $db->prepare("SELECT COUNT(*) as count FROM invoices WHERE user_id = ?");
    $stmt->bind_param("i", $user['id']);
    $stmt->execute();
    $count = $stmt->get_result()->fetch_assoc()['count'];
    $invoiceNumber = INVOICE_PREFIX . str_pad($count + 1, 4, '0', STR_PAD_LEFT);
} catch (Exception $e) {
    $invoiceNumber = INVOICE_PREFIX . '0001';
}
?>

<?php include '../includes/header.php'; ?>

<div class="container-fluid py-4">
    <div class="row">
        <!-- Sidebar -->
        <div class="col-lg-3 col-xl-2">
            <div class="card border-0 shadow-sm">
                <div class="card-body p-3">
                    <div class="d-flex align-items-center mb-4">
                        <div class="avatar bg-primary text-white rounded-circle me-3 d-flex align-items-center justify-content-center" style="width: 50px; height: 50px;">
                            <?php echo strtoupper(substr($user['name'], 0, 2)); ?>
                        </div>
                        <div>
                            <h6 class="mb-0 fw-bold"><?php echo htmlspecialchars($user['name']); ?></h6>
                            <small class="text-muted"><?php echo htmlspecialchars($user['email']); ?></small>
                        </div>
                    </div>
                    
                    <nav class="nav flex-column">
                        <a class="nav-link d-flex align-items-center py-2" href="dashboard.php">
                            <i class="bi bi-speedometer2 me-2"></i>Dashboard
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="invoices.php">
                            <i class="bi bi-receipt me-2"></i>Invoices
                        </a>
                        <a class="nav-link active d-flex align-items-center py-2" href="create-invoice.php">
                            <i class="bi bi-plus-circle me-2"></i>Create Invoice
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="clients.php">
                            <i class="bi bi-people me-2"></i>Clients
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="settings.php">
                            <i class="bi bi-gear me-2"></i>Settings
                        </a>
                    </nav>
                </div>
            </div>
        </div>
        
        <!-- Main Content -->
        <div class="col-lg-9 col-xl-10">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h1 class="h3 fw-bold mb-1">Create New Invoice</h1>
                    <p class="text-muted mb-0">Fill in the details below to create a professional invoice</p>
                </div>
                <a href="invoices.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left me-2"></i>Back to Invoices
                </a>
            </div>
            
            <form id="invoiceForm" action="../actions/create_invoice.php" method="POST" x-data="invoiceForm()">
                <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                
                <div class="row g-4">
                    <!-- Invoice Details -->
                    <div class="col-lg-8">
                        <div class="card border-0 shadow-sm">
                            <div class="card-header bg-white border-0 py-3">
                                <h5 class="mb-0 fw-bold">Invoice Details</h5>
                            </div>
                            <div class="card-body">
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <label for="invoice_number" class="form-label">Invoice Number</label>
                                        <input type="text" class="form-control" id="invoice_number" name="invoice_number" 
                                               value="<?php echo $invoiceNumber; ?>" readonly>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <label for="due_date" class="form-label">Due Date</label>
                                        <input type="date" class="form-control" id="due_date" name="due_date" 
                                               x-model="form.due_date" required>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <label for="client_name" class="form-label">Client Name</label>
                                        <input type="text" class="form-control" id="client_name" name="client_name" 
                                               x-model="form.client_name" placeholder="Enter client name" required>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <label for="client_email" class="form-label">Client Email</label>
                                        <input type="email" class="form-control" id="client_email" name="client_email" 
                                               x-model="form.client_email" placeholder="client@example.com" required>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Invoice Items -->
                        <div class="card border-0 shadow-sm mt-4">
                            <div class="card-header bg-white border-0 py-3">
                                <div class="d-flex justify-content-between align-items-center">
                                    <h5 class="mb-0 fw-bold">Invoice Items</h5>
                                    <button type="button" class="btn btn-outline-primary btn-sm" @click="addItem()">
                                        <i class="bi bi-plus-circle me-1"></i>Add Item
                                    </button>
                                </div>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table">
                                        <thead class="table-light">
                                            <tr>
                                                <th>Description</th>
                                                <th width="120">Quantity</th>
                                                <th width="150">Unit Price (<?php echo CURRENCY_SYMBOL; ?>)</th>
                                                <th width="150">Total (<?php echo CURRENCY_SYMBOL; ?>)</th>
                                                <th width="50"></th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <template x-for="(item, index) in form.items" :key="index">
                                                <tr>
                                                    <td>
                                                        <input type="text" class="form-control" 
                                                               :name="'items[' + index + '][description]'"
                                                               x-model="item.description" 
                                                               placeholder="Item description" required>
                                                    </td>
                                                    <td>
                                                        <input type="number" class="form-control" 
                                                               :name="'items[' + index + '][quantity]'"
                                                               x-model="item.quantity" 
                                                               @input="calculateItemTotal(index)"
                                                               min="1" step="1" required>
                                                    </td>
                                                    <td>
                                                        <input type="number" class="form-control" 
                                                               :name="'items[' + index + '][unit_price]'"
                                                               x-model="item.unit_price" 
                                                               @input="calculateItemTotal(index)"
                                                               min="0" step="0.01" required>
                                                    </td>
                                                    <td>
                                                        <input type="text" class="form-control" 
                                                               :value="formatCurrency(item.total)" readonly>
                                                        <input type="hidden" :name="'items[' + index + '][total]'" 
                                                               :value="item.total">
                                                    </td>
                                                    <td>
                                                        <button type="button" class="btn btn-outline-danger btn-sm" 
                                                                @click="removeItem(index)" x-show="form.items.length > 1">
                                                            <i class="bi bi-trash"></i>
                                                        </button>
                                                    </td>
                                                </tr>
                                            </template>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Invoice Summary -->
                    <div class="col-lg-4">
                        <div class="card border-0 shadow-sm sticky-top" style="top: 20px;">
                            <div class="card-header bg-white border-0 py-3">
                                <h5 class="mb-0 fw-bold">Invoice Summary</h5>
                            </div>
                            <div class="card-body">
                                <div class="d-flex justify-content-between mb-2">
                                    <span>Subtotal:</span>
                                    <span class="fw-bold" x-text="formatCurrency(form.subtotal)"></span>
                                </div>
                                
                                <div class="d-flex justify-content-between mb-2">
                                    <span>Tax (0%):</span>
                                    <span class="fw-bold" x-text="formatCurrency(form.tax)"></span>
                                </div>
                                
                                <hr>
                                
                                <div class="d-flex justify-content-between mb-3">
                                    <span class="fw-bold fs-5">Total:</span>
                                    <span class="fw-bold fs-5 text-primary" x-text="formatCurrency(form.total)"></span>
                                </div>
                                
                                <div class="d-grid gap-2">
                                    <button type="submit" class="btn btn-primary btn-lg" 
                                            :disabled="loading || !isFormValid()" 
                                            x-text="loading ? 'Creating...' : 'Create Invoice'">
                                    </button>
                                    
                                    <button type="button" class="btn btn-outline-secondary" @click="previewInvoice()" data-bs-toggle="modal" data-bs-target="#previewModal">
                                        <i class="bi bi-eye me-2"></i>Preview
                                    </button>
                                </div>
                                
                                <div class="mt-3 text-center">
                                    <small class="text-muted">
                                        <i class="bi bi-shield-check me-1"></i>
                                        Secure payment link will be generated
                                    </small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Preview Modal -->
<div class="modal fade" id="previewModal" tabindex="-1" aria-labelledby="previewModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="previewModalLabel">Invoice Preview</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="previewContent">
                <!-- Preview content will be inserted here -->
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<script>
function invoiceForm() {
    return {
        form: {
            client_name: '',
            client_email: '',
            due_date: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0], // 30 days from now
            items: [
                {
                    description: '',
                    quantity: 1,
                    unit_price: 0,
                    total: 0
                }
            ],
            subtotal: 0,
            tax: 0,
            total: 0
        },
        loading: false,
        
        addItem() {
            this.form.items.push({
                description: '',
                quantity: 1,
                unit_price: 0,
                total: 0
            });
        },
        
        removeItem(index) {
            if (this.form.items.length > 1) {
                this.form.items.splice(index, 1);
                this.calculateTotals();
            }
        },
        
        calculateItemTotal(index) {
            const item = this.form.items[index];
            item.total = parseFloat(item.quantity || 0) * parseFloat(item.unit_price || 0);
            this.calculateTotals();
        },
        
        calculateTotals() {
            this.form.subtotal = this.form.items.reduce((sum, item) => sum + parseFloat(item.total || 0), 0);
            this.form.tax = 0; // No tax for now
            this.form.total = this.form.subtotal + this.form.tax;
        },
        
        formatCurrency(amount) {
            return '<?php echo CURRENCY_SYMBOL; ?>' + parseFloat(amount || 0).toLocaleString('en-NG', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
        },
        
        isFormValid() {
            return this.form.client_name.trim() && 
                   this.form.client_email.trim() && 
                   this.form.due_date &&
                   this.form.items.every(item => item.description.trim() && item.quantity > 0 && item.unit_price >= 0) &&
                   this.form.total > 0;
        },
        
        previewInvoice() {
            if (!this.isFormValid()) {
                alert('Please fill in all required fields before previewing.');
                return;
            }
            
            const previewContent = `
                <div class="invoice-preview">
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <h4 class="text-primary">SwiftBill NG</h4>
                            <p class="text-muted">Professional Invoice</p>
                        </div>
                        <div class="col-md-6 text-end">
                            <h5>Invoice #<?php echo $invoiceNumber; ?></h5>
                            <p class="text-muted">Due: ${new Date(this.form.due_date).toLocaleDateString()}</p>
                        </div>
                    </div>
                    
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <h6>Bill To:</h6>
                            <p class="mb-1"><strong>${this.form.client_name}</strong></p>
                            <p class="text-muted">${this.form.client_email}</p>
                        </div>
                    </div>
                    
                    <div class="table-responsive mb-4">
                        <table class="table">
                            <thead class="table-light">
                                <tr>
                                    <th>Description</th>
                                    <th>Qty</th>
                                    <th>Price</th>
                                    <th>Total</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${this.form.items.map(item => `
                                    <tr>
                                        <td>${item.description}</td>
                                        <td>${item.quantity}</td>
                                        <td>${this.formatCurrency(item.unit_price)}</td>
                                        <td>${this.formatCurrency(item.total)}</td>
                                    </tr>
                                `).join('')}
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 offset-md-6">
                            <table class="table table-borderless">
                                <tr>
                                    <td>Subtotal:</td>
                                    <td class="text-end">${this.formatCurrency(this.form.subtotal)}</td>
                                </tr>
                                <tr>
                                    <td>Tax:</td>
                                    <td class="text-end">${this.formatCurrency(this.form.tax)}</td>
                                </tr>
                                <tr class="fw-bold border-top">
                                    <td>Total:</td>
                                    <td class="text-end text-primary">${this.formatCurrency(this.form.total)}</td>
                                </tr>
                            </table>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('previewContent').innerHTML = previewContent;
        },
        
        init() {
            this.calculateTotals();
            
            // Watch for changes in items
            this.$watch('form.items', () => {
                this.calculateTotals();
            }, { deep: true });
        }
    }
}
</script>

<?php include '../includes/footer.php'; ?>
