<?php
require_once '../includes/config.php';
require_once '../includes/auth.php';
require_once '../includes/db.php';

$auth = getAuth();
$page_title = 'Dashboard';

// Require login
$auth->requireLogin();

$user = $auth->getCurrentUser();
$db = getDB();

// Get dashboard statistics
try {
    // Total invoices
    $stmt = $db->prepare("SELECT COUNT(*) as total FROM invoices WHERE user_id = ?");
    $stmt->bind_param("i", $user['id']);
    $stmt->execute();
    $totalInvoices = $stmt->get_result()->fetch_assoc()['total'];
    
    // Paid invoices
    $stmt = $db->prepare("SELECT COUNT(*) as paid FROM invoices WHERE user_id = ? AND status = 'paid'");
    $stmt->bind_param("i", $user['id']);
    $stmt->execute();
    $paidInvoices = $stmt->get_result()->fetch_assoc()['paid'];
    
    // Unpaid invoices
    $stmt = $db->prepare("SELECT COUNT(*) as unpaid FROM invoices WHERE user_id = ? AND status = 'unpaid'");
    $stmt->bind_param("i", $user['id']);
    $stmt->execute();
    $unpaidInvoices = $stmt->get_result()->fetch_assoc()['unpaid'];
    
    // Total revenue
    $stmt = $db->prepare("SELECT SUM(total_amount) as revenue FROM invoices WHERE user_id = ? AND status = 'paid'");
    $stmt->bind_param("i", $user['id']);
    $stmt->execute();
    $totalRevenue = $stmt->get_result()->fetch_assoc()['revenue'] ?? 0;
    
    // Recent invoices
    $stmt = $db->prepare("SELECT * FROM invoices WHERE user_id = ? ORDER BY created_at DESC LIMIT 5");
    $stmt->bind_param("i", $user['id']);
    $stmt->execute();
    $recentInvoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    
} catch (Exception $e) {
    error_log("Dashboard error: " . $e->getMessage());
    $totalInvoices = $paidInvoices = $unpaidInvoices = $totalRevenue = 0;
    $recentInvoices = [];
}
?>

<?php include '../includes/header.php'; ?>

<div class="container-fluid py-4">
    <div class="row">
        <!-- Sidebar -->
        <div class="col-lg-3 col-xl-2">
            <div class="card border-0 shadow-sm">
                <div class="card-body p-3">
                    <div class="d-flex align-items-center mb-4">
                        <div class="avatar bg-primary text-white rounded-circle me-3 d-flex align-items-center justify-content-center" style="width: 50px; height: 50px;">
                            <?php echo strtoupper(substr($user['name'], 0, 2)); ?>
                        </div>
                        <div>
                            <h6 class="mb-0 fw-bold"><?php echo htmlspecialchars($user['name']); ?></h6>
                            <small class="text-muted"><?php echo htmlspecialchars($user['email']); ?></small>
                        </div>
                    </div>
                    
                    <nav class="nav flex-column">
                        <a class="nav-link active d-flex align-items-center py-2" href="dashboard.php">
                            <i class="bi bi-speedometer2 me-2"></i>Dashboard
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="invoices.php">
                            <i class="bi bi-receipt me-2"></i>Invoices
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="create-invoice.php">
                            <i class="bi bi-plus-circle me-2"></i>Create Invoice
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="clients.php">
                            <i class="bi bi-people me-2"></i>Clients
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="settings.php">
                            <i class="bi bi-gear me-2"></i>Settings
                        </a>
                    </nav>
                </div>
            </div>
        </div>
        
        <!-- Main Content -->
        <div class="col-lg-9 col-xl-10">
            <!-- Welcome Section -->
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h1 class="h3 fw-bold mb-1">Welcome back, <?php echo htmlspecialchars(explode(' ', $user['name'])[0]); ?>!</h1>
                    <p class="text-muted mb-0">Here's what's happening with your business today.</p>
                </div>
                <a href="create-invoice.php" class="btn btn-primary">
                    <i class="bi bi-plus-circle me-2"></i>Create Invoice
                </a>
            </div>
            
            <!-- Stats Cards -->
            <div class="row g-3 mb-4">
                <div class="col-sm-6 col-xl-3">
                    <div class="card border-0 shadow-sm">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-shrink-0">
                                    <div class="bg-primary bg-opacity-10 rounded p-3">
                                        <i class="bi bi-receipt text-primary fs-4"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1 ms-3">
                                    <div class="fw-bold fs-4"><?php echo number_format($totalInvoices); ?></div>
                                    <div class="text-muted small">Total Invoices</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-sm-6 col-xl-3">
                    <div class="card border-0 shadow-sm">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-shrink-0">
                                    <div class="bg-success bg-opacity-10 rounded p-3">
                                        <i class="bi bi-check-circle text-success fs-4"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1 ms-3">
                                    <div class="fw-bold fs-4"><?php echo number_format($paidInvoices); ?></div>
                                    <div class="text-muted small">Paid Invoices</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-sm-6 col-xl-3">
                    <div class="card border-0 shadow-sm">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-shrink-0">
                                    <div class="bg-warning bg-opacity-10 rounded p-3">
                                        <i class="bi bi-clock text-warning fs-4"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1 ms-3">
                                    <div class="fw-bold fs-4"><?php echo number_format($unpaidInvoices); ?></div>
                                    <div class="text-muted small">Pending Payment</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-sm-6 col-xl-3">
                    <div class="card border-0 shadow-sm">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-shrink-0">
                                    <div class="bg-info bg-opacity-10 rounded p-3">
                                        <i class="bi bi-currency-exchange text-info fs-4"></i>
                                    </div>
                                </div>
                                <div class="flex-grow-1 ms-3">
                                    <div class="fw-bold fs-4"><?php echo CURRENCY_SYMBOL . number_format($totalRevenue, 2); ?></div>
                                    <div class="text-muted small">Total Revenue</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Recent Invoices -->
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white border-0 py-3">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="mb-0 fw-bold">Recent Invoices</h5>
                        <a href="invoices.php" class="btn btn-outline-primary btn-sm">View All</a>
                    </div>
                </div>
                <div class="card-body p-0">
                    <?php if (empty($recentInvoices)): ?>
                        <div class="text-center py-5">
                            <i class="bi bi-receipt text-muted" style="font-size: 3rem;"></i>
                            <h5 class="mt-3 text-muted">No invoices yet</h5>
                            <p class="text-muted">Create your first invoice to get started</p>
                            <a href="create-invoice.php" class="btn btn-primary">
                                <i class="bi bi-plus-circle me-2"></i>Create Invoice
                            </a>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th>Invoice #</th>
                                        <th>Client</th>
                                        <th>Amount</th>
                                        <th>Status</th>
                                        <th>Date</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recentInvoices as $invoice): ?>
                                        <tr>
                                            <td>
                                                <span class="fw-bold"><?php echo htmlspecialchars($invoice['invoice_number']); ?></span>
                                            </td>
                                            <td><?php echo htmlspecialchars($invoice['client_name']); ?></td>
                                            <td class="fw-bold"><?php echo CURRENCY_SYMBOL . number_format($invoice['total_amount'], 2); ?></td>
                                            <td>
                                                <?php if ($invoice['status'] === 'paid'): ?>
                                                    <span class="badge bg-success">Paid</span>
                                                <?php elseif ($invoice['status'] === 'unpaid'): ?>
                                                    <span class="badge bg-warning">Unpaid</span>
                                                <?php else: ?>
                                                    <span class="badge bg-danger">Overdue</span>
                                                <?php endif; ?>
                                            </td>
                                            <td><?php echo date('M j, Y', strtotime($invoice['created_at'])); ?></td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <a href="view-invoice.php?id=<?php echo $invoice['id']; ?>" class="btn btn-outline-primary">
                                                        <i class="bi bi-eye"></i>
                                                    </a>
                                                    <?php if ($invoice['status'] === 'unpaid'): ?>
                                                        <a href="send-invoice.php?id=<?php echo $invoice['id']; ?>" class="btn btn-outline-success">
                                                            <i class="bi bi-send"></i>
                                                        </a>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
