<?php
session_start();
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/auth.php';

// Check if user is logged in
$auth = new Auth();
if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

// Get invoice ID from URL
$invoiceId = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$invoiceId) {
    $_SESSION['error_message'] = 'Invalid invoice ID.';
    header('Location: invoices.php');
    exit;
}

try {
    $db = Database::getInstance()->getConnection();
    $userId = $_SESSION['user_id'];
    
    // Get invoice details
    $stmt = $db->prepare("
        SELECT i.*, u.name as sender_name, u.email as sender_email
        FROM invoices i 
        JOIN users u ON i.user_id = u.id
        WHERE i.id = ? AND i.user_id = ?
    ");
    $stmt->bind_param("ii", $invoiceId, $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $invoice = $result->fetch_assoc();
    
    if (!$invoice) {
        $_SESSION['error_message'] = 'Invoice not found.';
        header('Location: invoices.php');
        exit;
    }
    
    // Get invoice items
    $stmt = $db->prepare("SELECT * FROM invoice_items WHERE invoice_id = ? ORDER BY id");
    $stmt->bind_param("i", $invoiceId);
    $stmt->execute();
    $items = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    
} catch (Exception $e) {
    error_log("Send invoice page error: " . $e->getMessage());
    $_SESSION['error_message'] = 'An error occurred. Please try again.';
    header('Location: invoices.php');
    exit;
}

$pageTitle = 'Send Invoice #' . $invoice['invoice_number'];
require_once '../includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <!-- Sidebar -->
        <nav class="col-md-3 col-lg-2 d-md-block bg-light sidebar collapse">
            <div class="position-sticky pt-3">
                <ul class="nav flex-column">
                    <li class="nav-item">
                        <a class="nav-link" href="dashboard.php">
                            <i class="fas fa-tachometer-alt"></i> Dashboard
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link active" href="invoices.php">
                            <i class="fas fa-file-invoice"></i> Invoices
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="create-invoice.php">
                            <i class="fas fa-plus"></i> Create Invoice
                        </a>
                    </li>
                </ul>
            </div>
        </nav>

        <!-- Main content -->
        <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">Send Invoice #<?= htmlspecialchars($invoice['invoice_number']) ?></h1>
                <div class="btn-toolbar mb-2 mb-md-0">
                    <a href="view-invoice.php?id=<?= $invoiceId ?>" class="btn btn-outline-secondary me-2">
                        <i class="fas fa-eye"></i> View Invoice
                    </a>
                    <a href="invoices.php" class="btn btn-outline-secondary">
                        <i class="fas fa-arrow-left"></i> Back to Invoices
                    </a>
                </div>
            </div>

            <div class="row">
                <div class="col-lg-8">
                    <!-- Invoice Preview -->
                    <div class="card mb-4">
                        <div class="card-header">
                            <h5 class="card-title mb-0">Invoice Preview</h5>
                        </div>
                        <div class="card-body">
                            <div class="invoice-preview">
                                <div class="row mb-4">
                                    <div class="col-6">
                                        <h4 class="text-primary">SwiftBill NG</h4>
                                        <p class="mb-1"><strong>From:</strong> <?= htmlspecialchars($invoice['sender_name']) ?></p>
                                        <p class="mb-1"><?= htmlspecialchars($invoice['sender_email']) ?></p>
                                    </div>
                                    <div class="col-6 text-end">
                                        <h4>Invoice #<?= htmlspecialchars($invoice['invoice_number']) ?></h4>
                                        <p class="mb-1"><strong>Date:</strong> <?= date('F j, Y', strtotime($invoice['created_at'])) ?></p>
                                        <p class="mb-1"><strong>Due:</strong> <?= date('F j, Y', strtotime($invoice['due_date'])) ?></p>
                                    </div>
                                </div>

                                <div class="row mb-4">
                                    <div class="col-6">
                                        <h6>Bill To:</h6>
                                        <p class="mb-1"><strong><?= htmlspecialchars($invoice['client_name']) ?></strong></p>
                                        <p class="mb-1"><?= htmlspecialchars($invoice['client_email']) ?></p>
                                        <?php if (!empty($invoice['client_address'])): ?>
                                            <p class="mb-1"><?= nl2br(htmlspecialchars($invoice['client_address'])) ?></p>
                                        <?php endif; ?>
                                    </div>
                                    <div class="col-6 text-end">
                                        <span class="badge bg-<?= $invoice['status'] === 'paid' ? 'success' : ($invoice['status'] === 'sent' ? 'warning' : 'secondary') ?> fs-6">
                                            <?= ucfirst($invoice['status']) ?>
                                        </span>
                                    </div>
                                </div>

                                <!-- Invoice Items -->
                                <div class="table-responsive mb-4">
                                    <table class="table table-bordered">
                                        <thead class="table-light">
                                            <tr>
                                                <th>Description</th>
                                                <th class="text-center">Qty</th>
                                                <th class="text-end">Unit Price</th>
                                                <th class="text-end">Total</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($items as $item): ?>
                                            <tr>
                                                <td><?= htmlspecialchars($item['item_name']) ?></td>
                                                <td class="text-center"><?= $item['quantity'] ?></td>
                                                <td class="text-end"><?= CURRENCY_SYMBOL . number_format($item['unit_price'], 2) ?></td>
                                                <td class="text-end"><?= CURRENCY_SYMBOL . number_format($item['total_price'], 2) ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                        <tfoot class="table-light">
                                            <tr>
                                                <th colspan="3" class="text-end">Total Amount:</th>
                                                <th class="text-end"><?= CURRENCY_SYMBOL . number_format($invoice['total_amount'], 2) ?></th>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>

                                <?php if (!empty($invoice['notes'])): ?>
                                <div class="mb-3">
                                    <h6>Notes:</h6>
                                    <p><?= nl2br(htmlspecialchars($invoice['notes'])) ?></p>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="col-lg-4">
                    <!-- Send Options -->
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0">Send Invoice</h5>
                        </div>
                        <div class="card-body">
                            <form action="../actions/send_invoice.php" method="POST" id="sendInvoiceForm">
                                <input type="hidden" name="csrf_token" value="<?= $auth->generateCsrfToken() ?>">
                                <input type="hidden" name="invoice_id" value="<?= $invoiceId ?>">
                                <input type="hidden" name="redirect_to" value="view-invoice.php?id=<?= $invoiceId ?>">

                                <div class="mb-3">
                                    <label for="client_email" class="form-label">Send To:</label>
                                    <input type="email" class="form-control" id="client_email" value="<?= htmlspecialchars($invoice['client_email']) ?>" readonly>
                                </div>

                                <div class="mb-3">
                                    <label class="form-label">Email Content Preview:</label>
                                    <div class="border rounded p-3 bg-light small">
                                        <strong>Subject:</strong> Invoice #<?= htmlspecialchars($invoice['invoice_number']) ?> from <?= htmlspecialchars($invoice['sender_name']) ?><br><br>
                                        <strong>Message:</strong><br>
                                        Hello <?= htmlspecialchars($invoice['client_name']) ?>,<br><br>
                                        You have received a new invoice from <?= htmlspecialchars($invoice['sender_name']) ?>.<br><br>
                                        <strong>Invoice Details:</strong><br>
                                        • Invoice Number: #<?= htmlspecialchars($invoice['invoice_number']) ?><br>
                                        • Amount: <?= CURRENCY_SYMBOL . number_format($invoice['total_amount'], 2) ?><br>
                                        • Due Date: <?= date('F j, Y', strtotime($invoice['due_date'])) ?><br><br>
                                        Click the "Pay Now" button in the email to make payment.
                                    </div>
                                </div>

                                <?php if ($invoice['status'] === 'sent' || $invoice['status'] === 'paid'): ?>
                                <div class="alert alert-info">
                                    <i class="fas fa-info-circle"></i>
                                    <?php if ($invoice['status'] === 'paid'): ?>
                                        This invoice has been paid.
                                    <?php else: ?>
                                        This invoice was already sent<?= !empty($invoice['sent_at']) ? ' on ' . date('M j, Y g:i A', strtotime($invoice['sent_at'])) : '' ?>.
                                    <?php endif; ?>
                                    Sending again will resend the email to the client.
                                </div>
                                <?php endif; ?>

                                <div class="d-grid gap-2">
                                    <button type="submit" class="btn btn-primary btn-lg" id="sendBtn">
                                        <i class="fas fa-paper-plane"></i> 
                                        <?= $invoice['status'] === 'draft' ? 'Send Invoice' : 'Resend Invoice' ?>
                                    </button>
                                    <a href="view-invoice.php?id=<?= $invoiceId ?>" class="btn btn-outline-secondary">
                                        Cancel
                                    </a>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Email Configuration Status -->
                    <div class="card mt-3">
                        <div class="card-body">
                            <h6 class="card-title">Email Configuration Status</h6>
                            <?php if (SMTP_USERNAME !== 'your_email@gmail.com' && !empty(SMTP_USERNAME)): ?>
                                <div class="text-success">
                                    <i class="fas fa-check-circle"></i> Email configured
                                </div>
                                <small class="text-muted">Emails will be sent via SMTP</small>
                            <?php else: ?>
                                <div class="text-warning">
                                    <i class="fas fa-exclamation-triangle"></i> Email not configured
                                </div>
                                <small class="text-muted">Update SMTP settings in config.php to send emails</small>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
document.getElementById('sendInvoiceForm').addEventListener('submit', function(e) {
    const sendBtn = document.getElementById('sendBtn');
    sendBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending...';
    sendBtn.disabled = true;
});
</script>

<?php require_once '../includes/footer.php'; ?>
