<?php
require_once '../includes/config.php';
require_once '../includes/auth.php';
require_once '../includes/db.php';

$auth = getAuth();
$page_title = 'View Invoice';

// Require login
$auth->requireLogin();

$user = $auth->getCurrentUser();
$db = getDB();

// Get invoice ID
$invoiceId = intval($_GET['id'] ?? 0);

if (!$invoiceId) {
    $_SESSION['error_message'] = 'Invalid invoice ID';
    header('Location: invoices.php');
    exit();
}

try {
    // Get invoice details
    $stmt = $db->prepare("SELECT * FROM invoices WHERE id = ? AND user_id = ?");
    $stmt->bind_param("ii", $invoiceId, $user['id']);
    $stmt->execute();
    $invoice = $stmt->get_result()->fetch_assoc();
    
    if (!$invoice) {
        $_SESSION['error_message'] = 'Invoice not found';
        header('Location: invoices.php');
        exit();
    }
    
    // Get invoice items
    $stmt = $db->prepare("SELECT * FROM invoice_items WHERE invoice_id = ? ORDER BY id");
    $stmt->bind_param("i", $invoiceId);
    $stmt->execute();
    $items = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    
} catch (Exception $e) {
    error_log("View invoice error: " . $e->getMessage());
    $_SESSION['error_message'] = 'Failed to load invoice';
    header('Location: invoices.php');
    exit();
}

$page_title = 'Invoice #' . $invoice['invoice_number'];
?>

<?php include '../includes/header.php'; ?>

<div class="container-fluid py-4">
    <div class="row">
        <!-- Sidebar -->
        <div class="col-lg-3 col-xl-2">
            <div class="card border-0 shadow-sm">
                <div class="card-body p-3">
                    <div class="d-flex align-items-center mb-4">
                        <div class="avatar bg-primary text-white rounded-circle me-3 d-flex align-items-center justify-content-center" style="width: 50px; height: 50px;">
                            <?php echo strtoupper(substr($user['name'], 0, 2)); ?>
                        </div>
                        <div>
                            <h6 class="mb-0 fw-bold"><?php echo htmlspecialchars($user['name']); ?></h6>
                            <small class="text-muted"><?php echo htmlspecialchars($user['email']); ?></small>
                        </div>
                    </div>
                    
                    <nav class="nav flex-column">
                        <a class="nav-link d-flex align-items-center py-2" href="dashboard.php">
                            <i class="bi bi-speedometer2 me-2"></i>Dashboard
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="invoices.php">
                            <i class="bi bi-receipt me-2"></i>Invoices
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="create-invoice.php">
                            <i class="bi bi-plus-circle me-2"></i>Create Invoice
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="clients.php">
                            <i class="bi bi-people me-2"></i>Clients
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="settings.php">
                            <i class="bi bi-gear me-2"></i>Settings
                        </a>
                    </nav>
                </div>
            </div>
        </div>
        
        <!-- Main Content -->
        <div class="col-lg-9 col-xl-10">
            <!-- Header -->
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h1 class="h3 fw-bold mb-1">Invoice #<?php echo htmlspecialchars($invoice['invoice_number']); ?></h1>
                    <p class="text-muted mb-0">
                        Created on <?php echo date('F j, Y', strtotime($invoice['created_at'])); ?>
                    </p>
                </div>
                <div class="d-flex gap-2">
                    <a href="invoices.php" class="btn btn-outline-secondary">
                        <i class="bi bi-arrow-left me-2"></i>Back
                    </a>
                    <a href="send-invoice.php?id=<?php echo $invoice['id']; ?>" class="btn btn-success">
                        <i class="bi bi-send me-2"></i>Send Invoice
                    </a>
                    <button class="btn btn-outline-primary" onclick="window.print()">
                        <i class="bi bi-printer me-2"></i>Print
                    </button>
                </div>
            </div>
            
            <!-- Invoice Status -->
            <div class="row mb-4">
                <div class="col-md-8">
                    <div class="card border-0 shadow-sm">
                        <div class="card-body">
                            <div class="row align-items-center">
                                <div class="col-md-6">
                                    <h5 class="mb-1">Payment Status</h5>
                                    <?php
                                    $statusClass = 'bg-secondary';
                                    $statusText = 'Unknown';
                                    $statusIcon = 'bi-question-circle';
                                    
                                    switch ($invoice['status']) {
                                        case 'paid':
                                            $statusClass = 'bg-success';
                                            $statusText = 'Paid';
                                            $statusIcon = 'bi-check-circle';
                                            break;
                                        case 'unpaid':
                                            $statusClass = 'bg-warning';
                                            $statusText = 'Unpaid';
                                            $statusIcon = 'bi-clock';
                                            break;
                                        case 'overdue':
                                            $statusClass = 'bg-danger';
                                            $statusText = 'Overdue';
                                            $statusIcon = 'bi-exclamation-triangle';
                                            break;
                                    }
                                    ?>
                                    <span class="badge <?php echo $statusClass; ?> fs-6">
                                        <i class="bi <?php echo $statusIcon; ?> me-1"></i><?php echo $statusText; ?>
                                    </span>
                                </div>
                                <div class="col-md-6 text-md-end">
                                    <h5 class="mb-1">Total Amount</h5>
                                    <span class="fs-4 fw-bold text-primary">
                                        <?php echo CURRENCY_SYMBOL . number_format($invoice['total_amount'], 2); ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card border-0 shadow-sm">
                        <div class="card-body text-center">
                            <h6 class="mb-2">Due Date</h6>
                            <p class="fs-5 fw-bold mb-0">
                                <?php echo date('M j, Y', strtotime($invoice['due_date'])); ?>
                            </p>
                            <?php if ($invoice['status'] === 'unpaid' && strtotime($invoice['due_date']) < time()): ?>
                                <small class="text-danger">
                                    <i class="bi bi-exclamation-triangle me-1"></i>Overdue
                                </small>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Invoice Details -->
            <div class="card border-0 shadow-sm">
                <div class="card-body p-4">
                    <!-- Invoice Header -->
                    <div class="row mb-5">
                        <div class="col-md-6">
                            <h2 class="text-primary fw-bold mb-3">
                                <i class="bi bi-receipt me-2"></i>SwiftBill NG
                            </h2>
                            <div class="text-muted">
                                <p class="mb-1">From: <strong><?php echo htmlspecialchars($user['name']); ?></strong></p>
                                <p class="mb-1"><?php echo htmlspecialchars($user['email']); ?></p>
                            </div>
                        </div>
                        <div class="col-md-6 text-md-end">
                            <h3 class="fw-bold mb-3">INVOICE</h3>
                            <div class="text-muted">
                                <p class="mb-1"><strong>Invoice #:</strong> <?php echo htmlspecialchars($invoice['invoice_number']); ?></p>
                                <p class="mb-1"><strong>Date:</strong> <?php echo date('M j, Y', strtotime($invoice['created_at'])); ?></p>
                                <p class="mb-1"><strong>Due Date:</strong> <?php echo date('M j, Y', strtotime($invoice['due_date'])); ?></p>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Bill To -->
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <h5 class="fw-bold mb-3">Bill To:</h5>
                            <div class="bg-light p-3 rounded">
                                <p class="mb-1 fw-bold"><?php echo htmlspecialchars($invoice['client_name']); ?></p>
                                <p class="mb-0 text-muted"><?php echo htmlspecialchars($invoice['client_email']); ?></p>
                            </div>
                        </div>
                        <?php if ($invoice['status'] === 'unpaid' && !empty($invoice['paystack_payment_link'])): ?>
                        <div class="col-md-6 text-md-end">
                            <h5 class="fw-bold mb-3">Payment Link:</h5>
                            <div class="bg-primary bg-opacity-10 p-3 rounded">
                                <button class="btn btn-primary btn-lg" onclick="copyPaymentLink('<?php echo htmlspecialchars($invoice['paystack_payment_link']); ?>')">
                                    <i class="bi bi-link-45deg me-2"></i>Copy Payment Link
                                </button>
                                <p class="small text-muted mt-2 mb-0">Share this link with your client for easy payment</p>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Invoice Items -->
                    <div class="table-responsive mb-4">
                        <table class="table table-borderless">
                            <thead class="table-light">
                                <tr>
                                    <th>Description</th>
                                    <th class="text-center">Quantity</th>
                                    <th class="text-end">Unit Price</th>
                                    <th class="text-end">Total</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($items as $item): ?>
                                    <tr>
                                        <td>
                                            <div class="fw-bold"><?php echo htmlspecialchars($item['item_name']); ?></div>
                                        </td>
                                        <td class="text-center"><?php echo number_format($item['quantity']); ?></td>
                                        <td class="text-end"><?php echo CURRENCY_SYMBOL . number_format($item['unit_price'], 2); ?></td>
                                        <td class="text-end fw-bold"><?php echo CURRENCY_SYMBOL . number_format($item['total_price'], 2); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Invoice Totals -->
                    <div class="row">
                        <div class="col-md-6"></div>
                        <div class="col-md-6">
                            <div class="table-responsive">
                                <table class="table table-borderless">
                                    <tr>
                                        <td class="text-end">Subtotal:</td>
                                        <td class="text-end fw-bold"><?php echo CURRENCY_SYMBOL . number_format($invoice['subtotal'], 2); ?></td>
                                    </tr>
                                    <tr>
                                        <td class="text-end">Tax:</td>
                                        <td class="text-end fw-bold"><?php echo CURRENCY_SYMBOL . number_format($invoice['tax_amount'], 2); ?></td>
                                    </tr>
                                    <tr class="border-top">
                                        <td class="text-end fs-5 fw-bold">Total:</td>
                                        <td class="text-end fs-4 fw-bold text-primary"><?php echo CURRENCY_SYMBOL . number_format($invoice['total_amount'], 2); ?></td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                    </div>
                    
                    <?php if ($invoice['status'] === 'paid' && $invoice['paid_at']): ?>
                        <div class="alert alert-success mt-4">
                            <div class="d-flex align-items-center">
                                <i class="bi bi-check-circle-fill fs-4 me-3"></i>
                                <div>
                                    <h6 class="mb-1">Payment Received</h6>
                                    <p class="mb-0">This invoice was paid on <?php echo date('F j, Y \a\t g:i A', strtotime($invoice['paid_at'])); ?></p>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function copyPaymentLink(link) {
    SwiftBill.copyToClipboard(link);
}

function sendInvoice(invoiceId) {
    if (confirm('Send this invoice to the client?')) {
        // TODO: Implement send invoice functionality
        SwiftBill.showToast('Invoice sent successfully!', 'success');
    }
}
</script>

<?php include '../includes/footer.php'; ?>
