<?php
require_once '../includes/config.php';
require_once '../includes/auth.php';
require_once '../includes/email.php';

$auth = getAuth();

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ../public/contact.php');
    exit();
}

// Verify CSRF token
if (!$auth->verifyCSRFToken($_POST['csrf_token'] ?? '')) {
    $_SESSION['error_message'] = 'Invalid request. Please try again.';
    header('Location: ../public/contact.php');
    exit();
}

// Sanitize and validate input
$name = $auth->sanitizeInput($_POST['name'] ?? '');
$email = $auth->sanitizeInput($_POST['email'] ?? '');
$subject = $auth->sanitizeInput($_POST['subject'] ?? '');
$message = $auth->sanitizeInput($_POST['message'] ?? '');

// Validation
$errors = [];

if (empty($name)) {
    $errors[] = 'Name is required';
}

if (empty($email) || !$auth->validateEmail($email)) {
    $errors[] = 'Valid email address is required';
}

if (empty($subject)) {
    $errors[] = 'Subject is required';
}

if (empty($message)) {
    $errors[] = 'Message is required';
} elseif (strlen($message) < 10) {
    $errors[] = 'Message must be at least 10 characters';
}

// If validation fails, redirect back with errors
if (!empty($errors)) {
    $_SESSION['error_message'] = implode('<br>', $errors);
    header('Location: ../public/contact.php');
    exit();
}

try {
    // Save message to database
    $db = getDB();
    $stmt = $db->prepare("INSERT INTO contact_messages (name, email, message) VALUES (?, ?, ?)");
    $fullMessage = "Subject: " . $subject . "\n\n" . $message;
    $stmt->bind_param("sss", $name, $email, $fullMessage);
    
    if ($stmt->execute()) {
        // Send email notification
        $emailSender = new EmailSender();
        $emailResult = $emailSender->sendContactMessage($name, $email, $fullMessage);
        
        if ($emailResult['success']) {
            $_SESSION['success_message'] = 'Thank you for your message! We\'ll get back to you within 24 hours.';
        } else {
            $_SESSION['success_message'] = 'Your message has been saved. We\'ll get back to you soon.';
        }
    } else {
        $_SESSION['error_message'] = 'Failed to send message. Please try again.';
    }
    
} catch (Exception $e) {
    error_log("Contact form error: " . $e->getMessage());
    $_SESSION['error_message'] = 'Failed to send message. Please try again.';
}

header('Location: ../public/contact.php');
exit();
?>
