<?php
require_once '../includes/config.php';
require_once '../includes/auth.php';
require_once '../includes/email.php';

$auth = getAuth();

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ../public/register.php');
    exit();
}

// Verify CSRF token
if (!$auth->verifyCSRFToken($_POST['csrf_token'] ?? '')) {
    $_SESSION['error_message'] = 'Invalid request. Please try again.';
    header('Location: ../public/register.php');
    exit();
}

// Sanitize and validate input
$fullName = $auth->sanitizeInput($_POST['full_name'] ?? '');
$email = $auth->sanitizeInput($_POST['email'] ?? '');
$password = $_POST['password'] ?? '';
$confirmPassword = $_POST['confirm_password'] ?? '';
$terms = isset($_POST['terms']);

// Validation
$errors = [];

if (empty($fullName)) {
    $errors[] = 'Full name is required';
}

if (empty($email) || !$auth->validateEmail($email)) {
    $errors[] = 'Valid email address is required';
}

if (empty($password)) {
    $errors[] = 'Password is required';
} elseif (!$auth->validatePassword($password)) {
    $errors[] = 'Password must be at least 8 characters with uppercase, lowercase, number, and special character';
}

if ($password !== $confirmPassword) {
    $errors[] = 'Passwords do not match';
}

if (!$terms) {
    $errors[] = 'You must agree to the terms and conditions';
}

// If validation fails, redirect back with errors
if (!empty($errors)) {
    $_SESSION['error_message'] = implode('<br>', $errors);
    header('Location: ../public/register.php');
    exit();
}

// Attempt registration
$result = $auth->register($fullName, $email, $password);

if ($result['success']) {
    // Send verification email
    $emailSender = new EmailSender();
    $verificationLink = SITE_URL . '/verify-email.php?token=' . $result['verification_token'];
    
    $emailResult = $emailSender->sendVerificationEmail($email, $fullName, $verificationLink);
    
    if ($emailResult['success']) {
        $_SESSION['success_message'] = 'Registration successful! Please check your email to verify your account.';
    } else {
        $_SESSION['success_message'] = 'Registration successful! However, we couldn\'t send the verification email. Please contact support.';
    }
    
    header('Location: ../public/login.php');
} else {
    $_SESSION['error_message'] = $result['message'];
    header('Location: ../public/register.php');
}
exit();
?>
