<?php
// Email functionality using PHPMailer
require_once __DIR__ . '/../vendor/phpmailer/phpmailer/src/Exception.php';
require_once __DIR__ . '/../vendor/phpmailer/phpmailer/src/PHPMailer.php';
require_once __DIR__ . '/../vendor/phpmailer/phpmailer/src/SMTP.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

class EmailSender {
    private $mailer;
    
    public function __construct() {
        $this->mailer = new PHPMailer(true);
        $this->configureMailer();
    }
    
    private function configureMailer() {
        try {
            // Server settings
            $this->mailer->isSMTP();
            $this->mailer->Host = SMTP_HOST;
            $this->mailer->SMTPAuth = true;
            $this->mailer->Username = SMTP_USERNAME;
            $this->mailer->Password = SMTP_PASSWORD;
        $this->mailer->SMTPSecure = 'ssl';
            $this->mailer->Port = SMTP_PORT;
            
            // Default sender
            $this->mailer->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
            
        } catch (Exception $e) {
            error_log("Email configuration error: " . $e->getMessage());
        }
    }
    
    public function sendVerificationEmail($email, $name, $verificationLink) {
        try {
            $this->mailer->clearAddresses();
            $this->mailer->addAddress($email, $name);
            
            $this->mailer->isHTML(true);
            $this->mailer->Subject = 'Verify Your SwiftBill NG Account';
            
            $this->mailer->Body = $this->getVerificationEmailTemplate($name, $verificationLink);
            
            $this->mailer->send();
            return ['success' => true, 'message' => 'Verification email sent successfully'];
            
        } catch (Exception $e) {
            error_log("Verification email error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to send verification email'];
        }
    }
    
    public function sendInvoiceEmail($clientEmail, $clientName, $invoiceData, $paymentLink) {
        try {
            $this->mailer->clearAddresses();
            $this->mailer->addAddress($clientEmail, $clientName);
            
            $this->mailer->isHTML(true);
            $this->mailer->Subject = 'Invoice #' . $invoiceData['invoice_number'] . ' from ' . $invoiceData['sender_name'];
            
            $htmlBody = $this->getInvoiceEmailTemplate($clientName, $invoiceData, $paymentLink);
            $this->mailer->Body = $htmlBody;
            $this->mailer->AltBody = strip_tags($htmlBody);
            
            $this->mailer->send();
            return ['success' => true, 'message' => 'Invoice email sent successfully'];
            
        } catch (Exception $e) {
            error_log("Invoice email error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to send invoice email'];
        }
    }
    
    public function sendPaymentConfirmationEmail($clientEmail, $clientName, $invoiceData) {
        try {
            $this->mailer->clearAddresses();
            $this->mailer->addAddress($clientEmail, $clientName);
            
            $this->mailer->isHTML(true);
            $this->mailer->Subject = 'Payment Confirmation - Invoice #' . $invoiceData['invoice_number'];
            
            $htmlBody = $this->getPaymentConfirmationTemplate($clientName, $invoiceData);
            $this->mailer->Body = $htmlBody;
            $this->mailer->AltBody = strip_tags($htmlBody);
            
            $this->mailer->send();
            return ['success' => true, 'message' => 'Payment confirmation sent successfully'];
            
        } catch (Exception $e) {
            error_log("Payment confirmation email error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to send payment confirmation'];
        }
    }
    
    public function sendContactMessage($name, $email, $message) {
        try {
            $this->mailer->clearAddresses();
            $this->mailer->addAddress(ADMIN_EMAIL, 'SwiftBill NG Admin');
            $this->mailer->addReplyTo($email, $name);
            
            $this->mailer->isHTML(true);
            $this->mailer->Subject = 'New Contact Message from ' . $name;
            
            $htmlBody = $this->getContactMessageTemplate($name, $email, $message);
            $this->mailer->Body = $htmlBody;
            $this->mailer->AltBody = strip_tags($htmlBody);
            
            $this->mailer->send();
            return ['success' => true, 'message' => 'Message sent successfully'];
            
        } catch (Exception $e) {
            error_log("Contact message error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to send message'];
        }
    }
    
    private function getVerificationEmailTemplate($name, $verificationLink) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <title>Verify Your Account</title>
        </head>
        <body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333;'>
            <div style='max-width: 600px; margin: 0 auto; padding: 20px;'>
                <div style='text-align: center; margin-bottom: 30px;'>
                    <h1 style='color: #0d6efd;'>SwiftBill NG</h1>
                </div>
                
                <h2>Welcome to SwiftBill NG, {$name}!</h2>
                
                <p>Thank you for registering with SwiftBill NG. To complete your registration and start creating professional invoices, please verify your email address by clicking the button below:</p>
                
                <div style='text-align: center; margin: 30px 0;'>
                    <a href='{$verificationLink}' style='background-color: #0d6efd; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; display: inline-block;'>Verify Email Address</a>
                </div>
                
                <p>If the button doesn't work, you can copy and paste this link into your browser:</p>
                <p style='word-break: break-all; color: #0d6efd;'>{$verificationLink}</p>
                
                <p>This verification link will expire in 24 hours for security reasons.</p>
                
                <hr style='margin: 30px 0;'>
                
                <p style='font-size: 14px; color: #666;'>
                    If you didn't create an account with SwiftBill NG, please ignore this email.
                </p>
                
                <div style='text-align: center; margin-top: 30px; font-size: 14px; color: #666;'>
                    <p>&copy; " . date('Y') . " SwiftBill NG. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>";
    }
    
    private function getInvoiceEmailTemplate($clientName, $invoiceData, $paymentLink) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <title>Invoice #{$invoiceData['invoice_number']}</title>
        </head>
        <body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333;'>
            <div style='max-width: 600px; margin: 0 auto; padding: 20px;'>
                <div style='text-align: center; margin-bottom: 30px;'>
                    <h1 style='color: #0d6efd;'>SwiftBill NG</h1>
                </div>
                
                <h2>Invoice #{$invoiceData['invoice_number']}</h2>
                
                <p>Hello {$clientName},</p>
                
                <p>You have received a new invoice from {$invoiceData['sender_name']}.</p>
                
                <div style='background-color: #f8f9fa; padding: 20px; border-radius: 5px; margin: 20px 0;'>
                    <h3 style='margin-top: 0;'>Invoice Details</h3>
                    <p><strong>Invoice Number:</strong> #{$invoiceData['invoice_number']}</p>
                    <p><strong>Amount:</strong> " . CURRENCY_SYMBOL . number_format($invoiceData['total_amount'], 2) . "</p>
                    <p><strong>Due Date:</strong> {$invoiceData['due_date']}</p>
                </div>
                
                <div style='text-align: center; margin: 30px 0;'>
                    <a href='{$paymentLink}' style='background-color: #28a745; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; display: inline-block;'>Pay Now</a>
                </div>
                
                <p>You can also view and pay this invoice by copying this link into your browser:</p>
                <p style='word-break: break-all; color: #0d6efd;'>{$paymentLink}</p>
                
                <hr style='margin: 30px 0;'>
                
                <div style='text-align: center; margin-top: 30px; font-size: 14px; color: #666;'>
                    <p>Powered by SwiftBill NG - Professional Invoicing Made Simple</p>
                    <p>&copy; " . date('Y') . " SwiftBill NG. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>";
    }
    
    private function getPaymentConfirmationTemplate($clientName, $invoiceData) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <title>Payment Confirmation</title>
        </head>
        <body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333;'>
            <div style='max-width: 600px; margin: 0 auto; padding: 20px;'>
                <div style='text-align: center; margin-bottom: 30px;'>
                    <h1 style='color: #0d6efd;'>SwiftBill NG</h1>
                </div>
                
                <div style='text-align: center; margin-bottom: 30px;'>
                    <div style='background-color: #28a745; color: white; padding: 20px; border-radius: 50%; width: 60px; height: 60px; margin: 0 auto; display: flex; align-items: center; justify-content: center; font-size: 30px;'>✓</div>
                    <h2 style='color: #28a745; margin-top: 20px;'>Payment Successful!</h2>
                </div>
                
                <p>Hello {$clientName},</p>
                
                <p>Thank you for your payment! We have successfully received your payment for Invoice #{$invoiceData['invoice_number']}.</p>
                
                <div style='background-color: #f8f9fa; padding: 20px; border-radius: 5px; margin: 20px 0;'>
                    <h3 style='margin-top: 0;'>Payment Details</h3>
                    <p><strong>Invoice Number:</strong> #{$invoiceData['invoice_number']}</p>
                    <p><strong>Amount Paid:</strong> " . CURRENCY_SYMBOL . number_format($invoiceData['total_amount'], 2) . "</p>
                    <p><strong>Payment Date:</strong> " . date('F j, Y') . "</p>
                    <p><strong>Status:</strong> <span style='color: #28a745; font-weight: bold;'>PAID</span></p>
                </div>
                
                <p>This serves as your payment receipt. Please keep this email for your records.</p>
                
                <hr style='margin: 30px 0;'>
                
                <div style='text-align: center; margin-top: 30px; font-size: 14px; color: #666;'>
                    <p>Powered by SwiftBill NG - Professional Invoicing Made Simple</p>
                    <p>&copy; " . date('Y') . " SwiftBill NG. All rights reserved.</p>
                </div>
            </div>
        </body>
        </html>";
    }
    
    private function getContactMessageTemplate($name, $email, $message) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <title>New Contact Message</title>
        </head>
        <body style='font-family: Arial, sans-serif; line-height: 1.6; color: #333;'>
            <div style='max-width: 600px; margin: 0 auto; padding: 20px;'>
                <h2>New Contact Message from SwiftBill NG</h2>
                
                <div style='background-color: #f8f9fa; padding: 20px; border-radius: 5px; margin: 20px 0;'>
                    <p><strong>Name:</strong> {$name}</p>
                    <p><strong>Email:</strong> {$email}</p>
                    <p><strong>Date:</strong> " . date('F j, Y g:i A') . "</p>
                </div>
                
                <div style='background-color: #fff; border: 1px solid #ddd; padding: 20px; border-radius: 5px;'>
                    <h3>Message:</h3>
                    <p>" . nl2br(htmlspecialchars($message)) . "</p>
                </div>
            </div>
        </body>
        </html>";
    }
}
?>
