<?php
// Paystack API Integration
require_once 'config.php';

class PaystackAPI {
    private $secretKey;
    private $publicKey;
    private $baseUrl = 'https://api.paystack.co';
    
    public function __construct() {
        $this->secretKey = PAYSTACK_SECRET_KEY;
        $this->publicKey = PAYSTACK_PUBLIC_KEY;
    }
    
    /**
     * Make HTTP request to Paystack API
     */
    private function makeRequest($endpoint, $method = 'GET', $data = null) {
        $url = $this->baseUrl . $endpoint;
        
        $headers = [
            'Authorization: Bearer ' . $this->secretKey,
            'Content-Type: application/json',
            'Cache-Control: no-cache'
        ];
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_SSL_VERIFYPEER => true
        ]);
        
        if ($method === 'POST' && $data) {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            error_log("Paystack API cURL error: " . $error);
            return ['success' => false, 'message' => 'Network error occurred'];
        }
        
        $decodedResponse = json_decode($response, true);
        
        if ($httpCode !== 200 && $httpCode !== 201) {
            error_log("Paystack API error: HTTP $httpCode - " . $response);
            return [
                'success' => false, 
                'message' => $decodedResponse['message'] ?? 'API request failed'
            ];
        }
        
        return [
            'success' => true,
            'data' => $decodedResponse
        ];
    }
    
    /**
     * Initialize a payment transaction
     */
    public function initializePayment($data) {
        $requiredFields = ['email', 'amount'];
        
        foreach ($requiredFields as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                return ['success' => false, 'message' => "Missing required field: $field"];
            }
        }
        
        // Ensure amount is in kobo (multiply by 100)
        if ($data['amount'] < 100) {
            $data['amount'] = $data['amount'] * 100;
        }
        
        $result = $this->makeRequest('/transaction/initialize', 'POST', $data);
        
        if ($result['success'] && $result['data']['status']) {
            return [
                'success' => true,
                'authorization_url' => $result['data']['data']['authorization_url'],
                'access_code' => $result['data']['data']['access_code'],
                'reference' => $result['data']['data']['reference']
            ];
        }
        
        return [
            'success' => false,
            'message' => $result['data']['message'] ?? 'Failed to initialize payment'
        ];
    }
    
    /**
     * Create a payment link (for invoices)
     */
    public function createPaymentLink($data) {
        // Use initialize payment for now, as payment links require different endpoint
        return $this->initializePayment($data);
    }
    
    /**
     * Verify a payment transaction
     */
    public function verifyPayment($reference) {
        if (empty($reference)) {
            return ['success' => false, 'message' => 'Reference is required'];
        }
        
        $result = $this->makeRequest("/transaction/verify/$reference");
        
        if ($result['success'] && $result['data']['status']) {
            $transactionData = $result['data']['data'];
            
            return [
                'success' => true,
                'status' => $transactionData['status'],
                'reference' => $transactionData['reference'],
                'amount' => $transactionData['amount'] / 100, // Convert from kobo
                'currency' => $transactionData['currency'],
                'paid_at' => $transactionData['paid_at'],
                'customer' => $transactionData['customer'],
                'metadata' => $transactionData['metadata'] ?? []
            ];
        }
        
        return [
            'success' => false,
            'message' => $result['data']['message'] ?? 'Failed to verify payment'
        ];
    }
    
    /**
     * Get transaction details
     */
    public function getTransaction($transactionId) {
        if (empty($transactionId)) {
            return ['success' => false, 'message' => 'Transaction ID is required'];
        }
        
        $result = $this->makeRequest("/transaction/$transactionId");
        
        if ($result['success'] && $result['data']['status']) {
            return [
                'success' => true,
                'data' => $result['data']['data']
            ];
        }
        
        return [
            'success' => false,
            'message' => $result['data']['message'] ?? 'Failed to get transaction'
        ];
    }
    
    /**
     * List transactions
     */
    public function listTransactions($params = []) {
        $queryString = http_build_query($params);
        $endpoint = '/transaction' . ($queryString ? '?' . $queryString : '');
        
        $result = $this->makeRequest($endpoint);
        
        if ($result['success'] && $result['data']['status']) {
            return [
                'success' => true,
                'data' => $result['data']['data'],
                'meta' => $result['data']['meta'] ?? []
            ];
        }
        
        return [
            'success' => false,
            'message' => $result['data']['message'] ?? 'Failed to list transactions'
        ];
    }
    
    /**
     * Verify webhook signature
     */
    public function verifyWebhookSignature($payload, $signature) {
        $computedSignature = hash_hmac('sha512', $payload, PAYSTACK_WEBHOOK_SECRET);
        return hash_equals($signature, $computedSignature);
    }
    
    /**
     * Process webhook event
     */
    public function processWebhookEvent($payload) {
        $event = json_decode($payload, true);
        
        if (!$event) {
            return ['success' => false, 'message' => 'Invalid JSON payload'];
        }
        
        $eventType = $event['event'] ?? '';
        $eventData = $event['data'] ?? [];
        
        switch ($eventType) {
            case 'charge.success':
                return $this->handleChargeSuccess($eventData);
                
            case 'charge.failed':
                return $this->handleChargeFailed($eventData);
                
            default:
                return ['success' => true, 'message' => 'Event type not handled'];
        }
    }
    
    /**
     * Handle successful charge webhook
     */
    private function handleChargeSuccess($data) {
        try {
            $reference = $data['reference'] ?? '';
            $amount = ($data['amount'] ?? 0) / 100; // Convert from kobo
            $metadata = $data['metadata'] ?? [];
            $invoiceId = $metadata['invoice_id'] ?? null;
            
            if (!$invoiceId) {
                return ['success' => false, 'message' => 'Invoice ID not found in metadata'];
            }
            
            // Update invoice status
            $db = getDB();
            $stmt = $db->prepare("UPDATE invoices SET status = 'paid', paid_at = NOW() WHERE id = ? AND status = 'unpaid'");
            $stmt->bind_param("i", $invoiceId);
            
            if ($stmt->execute() && $stmt->affected_rows > 0) {
                // Get invoice details for email
                $stmt = $db->prepare("SELECT * FROM invoices WHERE id = ?");
                $stmt->bind_param("i", $invoiceId);
                $stmt->execute();
                $invoice = $stmt->get_result()->fetch_assoc();
                
                if ($invoice) {
                    // Send payment confirmation email
                    $emailSender = new EmailSender();
                    $emailSender->sendPaymentConfirmationEmail(
                        $invoice['client_email'],
                        $invoice['client_name'],
                        $invoice
                    );
                }
                
                return ['success' => true, 'message' => 'Payment processed successfully'];
            }
            
            return ['success' => false, 'message' => 'Failed to update invoice status'];
            
        } catch (Exception $e) {
            error_log("Webhook processing error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Error processing webhook'];
        }
    }
    
    /**
     * Handle failed charge webhook
     */
    private function handleChargeFailed($data) {
        // Log failed payment for monitoring
        error_log("Payment failed: " . json_encode($data));
        return ['success' => true, 'message' => 'Failed payment logged'];
    }
    
    /**
     * Get public key for frontend
     */
    public function getPublicKey() {
        return $this->publicKey;
    }
}
?>
