<?php
require_once '../includes/config.php';
require_once '../includes/auth.php';
require_once '../includes/db.php';

$auth = getAuth();
$page_title = 'Invoices';

// Require login
$auth->requireLogin();

$user = $auth->getCurrentUser();
$db = getDB();

// Get filter parameters
$status = $_GET['status'] ?? 'all';
$search = $_GET['search'] ?? '';
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 10;
$offset = ($page - 1) * $limit;

// Build query conditions
$conditions = ["user_id = ?"];
$params = [$user['id']];
$types = "i";

if ($status !== 'all') {
    $conditions[] = "status = ?";
    $params[] = $status;
    $types .= "s";
}

if (!empty($search)) {
    $conditions[] = "(invoice_number LIKE ? OR client_name LIKE ? OR client_email LIKE ?)";
    $searchTerm = "%$search%";
    $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm]);
    $types .= "sss";
}

$whereClause = "WHERE " . implode(" AND ", $conditions);

try {
    // Get total count
    $countQuery = "SELECT COUNT(*) as total FROM invoices $whereClause";
    $stmt = $db->prepare($countQuery);
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $totalInvoices = $stmt->get_result()->fetch_assoc()['total'];
    
    // Get invoices
    $query = "SELECT * FROM invoices $whereClause ORDER BY created_at DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    $types .= "ii";
    
    $stmt = $db->prepare($query);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    
    $totalPages = ceil($totalInvoices / $limit);
    
} catch (Exception $e) {
    error_log("Invoices page error: " . $e->getMessage());
    $invoices = [];
    $totalInvoices = 0;
    $totalPages = 1;
}
?>

<?php include '../includes/header.php'; ?>

<div class="container-fluid py-4">
    <div class="row">
        <!-- Sidebar -->
        <div class="col-lg-3 col-xl-2">
            <div class="card border-0 shadow-sm">
                <div class="card-body p-3">
                    <div class="d-flex align-items-center mb-4">
                        <div class="avatar bg-primary text-white rounded-circle me-3 d-flex align-items-center justify-content-center" style="width: 50px; height: 50px;">
                            <?php echo strtoupper(substr($user['name'], 0, 2)); ?>
                        </div>
                        <div>
                            <h6 class="mb-0 fw-bold"><?php echo htmlspecialchars($user['name']); ?></h6>
                            <small class="text-muted"><?php echo htmlspecialchars($user['email']); ?></small>
                        </div>
                    </div>
                    
                    <nav class="nav flex-column">
                        <a class="nav-link d-flex align-items-center py-2" href="dashboard.php">
                            <i class="bi bi-speedometer2 me-2"></i>Dashboard
                        </a>
                        <a class="nav-link active d-flex align-items-center py-2" href="invoices.php">
                            <i class="bi bi-receipt me-2"></i>Invoices
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="create-invoice.php">
                            <i class="bi bi-plus-circle me-2"></i>Create Invoice
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="clients.php">
                            <i class="bi bi-people me-2"></i>Clients
                        </a>
                        <a class="nav-link d-flex align-items-center py-2" href="settings.php">
                            <i class="bi bi-gear me-2"></i>Settings
                        </a>
                    </nav>
                </div>
            </div>
        </div>
        
        <!-- Main Content -->
        <div class="col-lg-9 col-xl-10">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h1 class="h3 fw-bold mb-1">Invoices</h1>
                    <p class="text-muted mb-0">Manage all your invoices in one place</p>
                </div>
                <a href="create-invoice.php" class="btn btn-primary">
                    <i class="bi bi-plus-circle me-2"></i>Create Invoice
                </a>
            </div>
            
            <!-- Filters -->
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-body">
                    <form method="GET" class="row g-3 align-items-end">
                        <div class="col-md-3">
                            <label for="status" class="form-label">Status</label>
                            <select class="form-select" id="status" name="status">
                                <option value="all" <?php echo $status === 'all' ? 'selected' : ''; ?>>All Invoices</option>
                                <option value="unpaid" <?php echo $status === 'unpaid' ? 'selected' : ''; ?>>Unpaid</option>
                                <option value="paid" <?php echo $status === 'paid' ? 'selected' : ''; ?>>Paid</option>
                                <option value="overdue" <?php echo $status === 'overdue' ? 'selected' : ''; ?>>Overdue</option>
                            </select>
                        </div>
                        
                        <div class="col-md-6">
                            <label for="search" class="form-label">Search</label>
                            <input type="text" class="form-control" id="search" name="search" 
                                   value="<?php echo htmlspecialchars($search); ?>" 
                                   placeholder="Search by invoice number, client name, or email">
                        </div>
                        
                        <div class="col-md-3">
                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-outline-primary">
                                    <i class="bi bi-search me-1"></i>Filter
                                </button>
                                <a href="invoices.php" class="btn btn-outline-secondary">
                                    <i class="bi bi-x-circle me-1"></i>Clear
                                </a>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
            
            <!-- Invoices Table -->
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white border-0 py-3">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="mb-0 fw-bold">
                            <?php echo number_format($totalInvoices); ?> Invoice<?php echo $totalInvoices !== 1 ? 's' : ''; ?>
                        </h5>
                        <div class="d-flex gap-2">
                            <button class="btn btn-outline-secondary btn-sm">
                                <i class="bi bi-download me-1"></i>Export
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body p-0">
                    <?php if (empty($invoices)): ?>
                        <div class="text-center py-5">
                            <i class="bi bi-receipt text-muted" style="font-size: 3rem;"></i>
                            <h5 class="mt-3 text-muted">No invoices found</h5>
                            <p class="text-muted">
                                <?php if (!empty($search) || $status !== 'all'): ?>
                                    Try adjusting your filters or search terms.
                                <?php else: ?>
                                    Create your first invoice to get started.
                                <?php endif; ?>
                            </p>
                            <?php if (empty($search) && $status === 'all'): ?>
                                <a href="create-invoice.php" class="btn btn-primary">
                                    <i class="bi bi-plus-circle me-2"></i>Create Invoice
                                </a>
                            <?php endif; ?>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th>Invoice #</th>
                                        <th>Client</th>
                                        <th>Amount</th>
                                        <th>Status</th>
                                        <th>Due Date</th>
                                        <th>Created</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($invoices as $invoice): ?>
                                        <tr>
                                            <td>
                                                <span class="fw-bold"><?php echo htmlspecialchars($invoice['invoice_number']); ?></span>
                                            </td>
                                            <td>
                                                <div>
                                                    <div class="fw-bold"><?php echo htmlspecialchars($invoice['client_name']); ?></div>
                                                    <small class="text-muted"><?php echo htmlspecialchars($invoice['client_email']); ?></small>
                                                </div>
                                            </td>
                                            <td class="fw-bold"><?php echo CURRENCY_SYMBOL . number_format($invoice['total_amount'], 2); ?></td>
                                            <td>
                                                <?php
                                                $statusClass = 'bg-secondary';
                                                $statusText = 'Unknown';
                                                
                                                switch ($invoice['status']) {
                                                    case 'paid':
                                                        $statusClass = 'bg-success';
                                                        $statusText = 'Paid';
                                                        break;
                                                    case 'unpaid':
                                                        $statusClass = 'bg-warning';
                                                        $statusText = 'Unpaid';
                                                        break;
                                                    case 'overdue':
                                                        $statusClass = 'bg-danger';
                                                        $statusText = 'Overdue';
                                                        break;
                                                }
                                                ?>
                                                <span class="badge <?php echo $statusClass; ?>"><?php echo $statusText; ?></span>
                                            </td>
                                            <td><?php echo date('M j, Y', strtotime($invoice['due_date'])); ?></td>
                                            <td><?php echo date('M j, Y', strtotime($invoice['created_at'])); ?></td>
                                            <td>
                                                <div class="btn-group btn-group-sm" role="group">
                                                    <a href="view-invoice.php?id=<?= $invoice['id'] ?>" class="btn btn-outline-primary" title="View">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <a href="send-invoice.php?id=<?= $invoice['id'] ?>" class="btn btn-outline-success" title="Send">
                                                        <i class="fas fa-paper-plane"></i>
                                                    </a>
                                                    <button class="btn btn-outline-danger" title="Delete" onclick="deleteInvoice(<?= $invoice['id'] ?>)">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                            <li><a class="dropdown-item" href="edit-invoice.php?id=<?php echo $invoice['id']; ?>">
                                                                <i class="bi bi-pencil me-2"></i>Edit
                                                            </a></li>
                                                            <li><a class="dropdown-item" href="duplicate-invoice.php?id=<?php echo $invoice['id']; ?>">
                                                                <i class="bi bi-files me-2"></i>Duplicate
                                                            </a></li>
                                                            <li><hr class="dropdown-divider"></li>
                                                            <li><a class="dropdown-item text-danger" href="#" 
                                                                   onclick="deleteInvoice(<?php echo $invoice['id']; ?>)">
                                                                <i class="bi bi-trash me-2"></i>Delete
                                                            </a></li>
                                                        </ul>
                                                    </div>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <!-- Pagination -->
                        <?php if ($totalPages > 1): ?>
                            <div class="card-footer bg-white border-0">
                                <nav aria-label="Invoice pagination">
                                    <ul class="pagination justify-content-center mb-0">
                                        <?php if ($page > 1): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo $page - 1; ?>&status=<?php echo $status; ?>&search=<?php echo urlencode($search); ?>">
                                                    <i class="bi bi-chevron-left"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                        
                                        <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                            <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                                <a class="page-link" href="?page=<?php echo $i; ?>&status=<?php echo $status; ?>&search=<?php echo urlencode($search); ?>">
                                                    <?php echo $i; ?>
                                                </a>
                                            </li>
                                        <?php endfor; ?>
                                        
                                        <?php if ($page < $totalPages): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo $page + 1; ?>&status=<?php echo $status; ?>&search=<?php echo urlencode($search); ?>">
                                                    <i class="bi bi-chevron-right"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </nav>
                            </div>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function deleteInvoice(invoiceId) {
    if (confirm('Are you sure you want to delete this invoice? This action cannot be undone.')) {
        // TODO: Implement delete functionality
        alert('Delete functionality will be implemented soon.');
    }
}
</script>

<?php include '../includes/footer.php'; ?>
